<?php

namespace Tobuli\Helpers\Payments\Gateways;


use Carbon\Carbon;
use Tobuli\Entities\Subscription;

abstract class PaymentGateway
{
    public function storeConfig($request, $gateway)
    {
        settings('payments.gateways.' . $gateway, $request->active ? 1 : 0);
        settings('payments.' . $gateway,
            array_merge(settings('payments.' . $gateway), $request->except('_token', 'active'))
        );
    }

    public function renewSubscription($subscription)
    {
        $expiration_date = $this->getExpirationDate(
            $subscription->user,
            $subscription->billing_plan
        );

        $subscription->user->update([
            'subscription_expiration' => $expiration_date,
        ]);

        $subscription->update([
            'active'          => true,
            'expiration_date' => $expiration_date,
        ]);
    }

    protected function gatewayName()
    {
        $name = (new \ReflectionClass($this))->getShortName();
        $name = str_replace('Gateway', '', $name);

        return snake_case($name);
    }

    protected function storeSubscription($user, $plan, $gateway_id)
    {
        return Subscription::create([
            'user_id'         => $user->id,
            'billing_plan_id' => $plan->id,
            'gateway'         => $this->gatewayName(),
            'gateway_id'      => $gateway_id,
        ]);
    }

    protected function activateSubscription($gateway_id, $options = [])
    {
        if ( ! $subscription = Subscription::where('gateway_id', $gateway_id)->first()) {
            throw new \Exception('Subscription not found for activation!');
        }

        $expiration_date = $this->getExpirationDate(
            $user = $subscription->user,
            $plan = $subscription->billing_plan
        );

        $user->update([
            'billing_plan_id'         => $plan->id,
            'devices_limit'           => $plan->objects,
            'subscription_expiration' => $expiration_date,
        ]);

        $subscription->update([
                'active'          => true,
                'expiration_date' => $expiration_date,
            ] + $options);
    }

    private function getExpirationDate($user, $plan)
    {
        if ( ! $user->isExpired() && $user->billing_plan_id == $plan->id) {
            $expiration_date = Carbon::createFromTimeString($user->subscription_expiration);
        } else {
            $expiration_date = Carbon::now();
        }

        switch ($plan->duration_type) {
            case 'days':
                $expiration_date->addDays($plan->duration_value);
                break;
            case 'months':
                $expiration_date->addMonths($plan->duration_value);
                break;
            case 'years':
                $expiration_date->addYears($plan->duration_value);
                break;
        }

        return $expiration_date->toDateTimeString();
    }
}