<?php namespace ModalHelpers;

use Facades\Repositories\EventRepo;
use Tobuli\Entities\Event;
use Formatter;

class EventModalHelper extends ModalHelper {

    public function lookup($data)
    {
        $this->checkException('events', 'view');

        $query = Event::with('device', 'geofence')
            ->select('events.*')
            ->orderBy('events.id', 'desc')
            ->where('events.deleted', 0);

        if ( ! empty($data['user_id']))
            $query->where('events.user_id', $data['user_id']);

        if ( ! empty($data['device_id']))
            $query->where('events.device_id', $data['device_id']);

        if ( ! empty($data['type']))
            $query->where('events.type', $data['type']);

        if ( ! empty($data['date_from']))
            $query->where('events.time', '>=', $data['date_from']);

        if ( ! empty($data['date_to']))
            $query->where('events.time', '<=', $data['date_to']);

        if ( ! empty($data['search'])) {
            $query->leftJoin('devices', 'events.device_id', '=', 'devices.id');

            $query->where(function ($q) use ($data) {
                $q->where('events.message', 'like', '%' . $data['search'] . '%');
                $q->orWhere('devices.name', 'like', '%' . $data['search'] . '%');
            });
        }

        $events = $query->paginate(30);

        $events->getCollection()->transform(function (Event $event)
        {
            $event->time     = Formatter::time()->convert($event->time);
            $event->speed    = Formatter::speed()->format($event->speed);
            $event->altitude = Formatter::altitude()->format($event->altitude);

            if ($device = $event->device) {
                $event->device_name = $event->device->name;
                unset($event->device->traccar);
            }

            if ($geofence = $event->geofence) {
                unset($event->geofence);

                $event->geofence = [
                    'id' => $geofence->id,
                    'name' => $geofence->name,
                ];
            }

            return $event;
        });

        if ($this->api)
            return collect(['url' => route('api.get_events')])->merge($events);

        return $events;
    }

    public function search($search, $device_id = null)
    {
        $this->checkException('events', 'view');

        $events = EventRepo::whereUserIdWithAttributes($this->user->id, $search, $device_id);

        foreach ($events as &$event) {
            $event->time = Formatter::time()->convert($event->time);

            $event->speed = round($this->user->unit_of_distance == 'mi' ? kilometersToMiles($event->speed) : $event->speed);
            $event->altitude = round($this->user->unit_of_altitude == 'ft' ? metersToFeets($event->altitude) : $event->altitude);
        }

        if ($this->api) {
            $events = $events->toArray();

            foreach ($events['data'] as &$event) {
                if(!empty($event['geofence']))
                    unset($event['geofence']);
            }
            $events['url'] = route('api.get_events');
        }

        return $events;
    }

    public function destroy($id = null)
    {
        $this->checkException('events', 'clean');

        $filter = [
            'id'      => $id,
            'user_id' => $this->user->id,
            'deleted' => 0
        ];

        if (empty($filter['id']))
            unset($filter['id']);

        EventRepo::deleteWhere($filter);
    }
}